// Configuration
const CONFIG = {
  // Production API URL
  API_BASE_URL: 'https://myrecipebook.net',
  // Development API URL (uncomment for local testing)
  // API_BASE_URL: 'http://localhost:5000',
};

// DOM Elements
const pageTitle = document.getElementById('pageTitle');
const pageUrl = document.getElementById('pageUrl');
const pageInfoCard = document.getElementById('pageInfoCard');
const extractBtn = document.getElementById('extractBtn');
const saveBtn = document.getElementById('saveBtn');
const editBtn = document.getElementById('editBtn');
const status = document.getElementById('status');
const mainContent = document.getElementById('mainContent');
const postExtractActions = document.getElementById('postExtractActions');
const settingsLink = document.getElementById('settingsLink');

// Tab elements
const tabCurrent = document.getElementById('tabCurrent');
const tabPaste = document.getElementById('tabPaste');
const tabContentCurrent = document.getElementById('tabContentCurrent');
const tabContentPaste = document.getElementById('tabContentPaste');
const pasteUrl = document.getElementById('pasteUrl');
const pasteFromClipboard = document.getElementById('pasteFromClipboard');

// Preview elements
const previewSection = document.getElementById('previewSection');
const previewImage = document.getElementById('previewImage');
const previewImagePlaceholder = document.getElementById('previewImagePlaceholder');
const previewTitle = document.getElementById('previewTitle');
const previewDescription = document.getElementById('previewDescription');
const previewSource = document.getElementById('previewSource');
const previewCategories = document.getElementById('previewCategories');
const extractingOverlay = document.getElementById('extractingOverlay');

// State
let activeTab = 'current';
let extractedData = null;
let currentPageData = {
  title: '',
  url: '',
  description: '',
  imageUrl: '',
  source: 'generic',
  confidence: 'low'
};

// Initialize popup
document.addEventListener('DOMContentLoaded', async () => {
  await loadCurrentPageInfo();
  setupEventListeners();
  await checkLoginStatus();
});

// Setup event listeners
function setupEventListeners() {
  extractBtn.addEventListener('click', handleExtract);
  saveBtn.addEventListener('click', handleSaveRecipe);
  editBtn.addEventListener('click', handleEditRecipe);
  
  settingsLink.addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: `${CONFIG.API_BASE_URL}/settings/sharing` });
  });

  // Tab switching
  tabCurrent.addEventListener('click', () => switchTab('current'));
  tabPaste.addEventListener('click', () => switchTab('paste'));

  // Paste URL input
  pasteUrl.addEventListener('input', handlePasteUrlInput);
  pasteUrl.addEventListener('paste', (e) => {
    setTimeout(handlePasteUrlInput, 50);
  });

  // Paste from clipboard button
  pasteFromClipboard.addEventListener('click', async () => {
    try {
      const text = await navigator.clipboard.readText();
      if (text && (text.startsWith('http://') || text.startsWith('https://'))) {
        pasteUrl.value = text;
        handlePasteUrlInput();
      }
    } catch (err) {
      console.log('Could not read clipboard');
    }
  });
}

// Switch between tabs
function switchTab(tab) {
  activeTab = tab;
  
  // Update tab buttons
  tabCurrent.classList.toggle('active', tab === 'current');
  tabPaste.classList.toggle('active', tab === 'paste');
  
  // Update tab content
  tabContentCurrent.style.display = tab === 'current' ? 'block' : 'none';
  tabContentPaste.style.display = tab === 'paste' ? 'block' : 'none';
  
  // Reset preview when switching tabs
  if (extractedData) {
    resetToInitialState();
  }
  
  // Update button text
  const btnText = extractBtn.querySelector('.btn-text');
  if (btnText) {
    btnText.textContent = 'חלץ מתכון';
  }
}

// Handle paste URL input
function handlePasteUrlInput() {
  const url = pasteUrl.value.trim();
  
  if (url && isValidUrl(url)) {
    // Update current page data for extraction
    currentPageData.url = url;
    currentPageData.title = '';
    currentPageData.source = detectSource(url);
    
    // Reset preview if exists
    if (extractedData) {
      resetToInitialState();
    }
  }
}

// Reset to initial state
function resetToInitialState() {
  extractedData = null;
  previewSection.style.display = 'none';
  postExtractActions.style.display = 'none';
  mainContent.style.display = 'block';
  extractBtn.disabled = false;
  extractBtn.classList.remove('loading');
  
  const btnText = extractBtn.querySelector('.btn-text');
  if (btnText) {
    btnText.textContent = 'חלץ מתכון';
  }
}

// Validate URL
function isValidUrl(string) {
  try {
    new URL(string);
    return string.startsWith('http://') || string.startsWith('https://');
  } catch (_) {
    return false;
  }
}

// Detect source from URL
function detectSource(url) {
  if (url.includes('facebook.com') || url.includes('fb.com') || url.includes('fb.watch')) return 'facebook';
  if (url.includes('instagram.com')) return 'instagram';
  if (url.includes('tiktok.com')) return 'tiktok';
  return 'generic';
}

// Load current tab's information with content script enhancement
async function loadCurrentPageInfo() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab) {
      pageTitle.textContent = 'שגיאה בטעינת מידע';
      return;
    }
    
    // Basic info from tab
    currentPageData.title = tab.title || 'ללא כותרת';
    currentPageData.url = tab.url || '';
    
    // Show loading state
    pageTitle.textContent = 'מחלץ מידע מהדף...';
    pageUrl.textContent = truncateUrl(currentPageData.url, 50);
    
    // Try to get enhanced data from content script
    const isSocialMedia = isSocialMediaUrl(tab.url);
    
    if (isSocialMedia && tab.id) {
      try {
        const response = await chrome.tabs.sendMessage(tab.id, { action: 'getPageContent' });
        
        if (response) {
          if (response.url) currentPageData.url = response.url;
          if (response.title && response.title !== 'Facebook' && !response.title.includes('| Facebook')) {
            currentPageData.title = response.title;
          }
          if (response.description) currentPageData.description = response.description;
          if (response.imageUrl) currentPageData.imageUrl = response.imageUrl;
          currentPageData.source = response.source || 'generic';
          currentPageData.confidence = response.confidence || 'low';
        }
      } catch (err) {
        try {
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            files: ['content.js']
          });
          
          await new Promise(resolve => setTimeout(resolve, 200));
          
          const response = await chrome.tabs.sendMessage(tab.id, { action: 'getPageContent' });
          if (response) {
            if (response.url) currentPageData.url = response.url;
            if (response.title && response.title !== 'Facebook' && !response.title.includes('| Facebook')) {
              currentPageData.title = response.title;
            }
            if (response.description) currentPageData.description = response.description;
            if (response.imageUrl) currentPageData.imageUrl = response.imageUrl;
            currentPageData.source = response.source || 'generic';
            currentPageData.confidence = response.confidence || 'low';
          }
        } catch (injectErr) {
          console.log('Could not inject content script:', injectErr);
        }
      }
    }
    
    // Update UI with final data
    updatePageInfoUI();
    
  } catch (error) {
    console.error('Error loading page info:', error);
    pageTitle.textContent = 'שגיאה בטעינת מידע';
  }
}

// Update the page info display
function updatePageInfoUI() {
  let displayTitle = currentPageData.title;
  
  // Remove common suffixes
  displayTitle = displayTitle
    .replace(/\s*\|\s*Facebook.*$/i, '')
    .replace(/\s*\|\s*Instagram.*$/i, '')
    .replace(/\s*-\s*TikTok.*$/i, '')
    .replace(/\s*on\s*TikTok$/i, '')
    .trim();
  
  // If title is too generic, use description preview
  const genericTitles = ['Facebook', 'Instagram', 'TikTok', 'Watch', 'Video'];
  if (genericTitles.includes(displayTitle)) {
    if (currentPageData.description) {
      displayTitle = currentPageData.description.substring(0, 80) + (currentPageData.description.length > 80 ? '...' : '');
    }
  }
  
  pageTitle.textContent = displayTitle || 'ללא כותרת';
  pageUrl.textContent = truncateUrl(currentPageData.url, 50);
  
  // Show source badge
  if (currentPageData.source !== 'generic') {
    const sourceLabels = {
      'facebook': '📘 פייסבוק',
      'instagram': '📷 אינסטגרם',
      'tiktok': '🎵 טיקטוק'
    };
    
    const badge = sourceLabels[currentPageData.source];
    if (badge && pageInfoCard) {
      const existingBadge = pageInfoCard.querySelector('.source-badge');
      if (existingBadge) existingBadge.remove();
      
      const badgeEl = document.createElement('div');
      badgeEl.className = 'source-badge';
      badgeEl.textContent = badge;
      badgeEl.style.cssText = `
        display: inline-block;
        padding: 4px 10px;
        background: linear-gradient(135deg, var(--primary-light), var(--primary));
        color: white;
        border-radius: 20px;
        font-size: 11px;
        font-weight: 600;
        margin-top: 8px;
      `;
      pageInfoCard.appendChild(badgeEl);
    }
  }
}

// Check if URL is a social media site
function isSocialMediaUrl(url) {
  if (!url) return false;
  const socialDomains = ['facebook.com', 'fb.com', 'fb.watch', 'instagram.com', 'tiktok.com'];
  return socialDomains.some(domain => url.includes(domain));
}

// Truncate URL for display
function truncateUrl(url, maxLength) {
  if (!url) return '';
  if (url.length <= maxLength) return url;
  return url.substring(0, maxLength) + '...';
}

// Check if user is logged in
async function checkLoginStatus() {
  try {
    console.log('Checking login status at:', `${CONFIG.API_BASE_URL}/api/user`);
    
    const response = await fetch(`${CONFIG.API_BASE_URL}/api/user`, {
      credentials: 'include'
    });
    
    console.log('Login check response:', response.status);
    
    if (!response.ok) {
      console.log('User not logged in, showing login prompt');
      showLoginPrompt();
    } else {
      const userData = await response.json();
      console.log('Logged in as:', userData.name);
    }
  } catch (error) {
    console.error('Error checking login status:', error);
    // If network error, still show login prompt since user might need to login
    showLoginPrompt('לא ניתן להתחבר לשרת. בדוק שאתה מחובר לאינטרנט');
  }
}

// Show login prompt
function showLoginPrompt(message = 'יש להתחבר כדי לשמור מתכונים') {
  mainContent.innerHTML = `
    <div class="login-prompt">
      <p>${message}</p>
      <button class="btn btn-login" id="loginBtn">
        <span class="btn-icon">🔐</span>
        <span class="btn-text">התחבר לחשבון</span>
      </button>
      <p class="login-hint">
        <small>פתח את <a href="${CONFIG.API_BASE_URL}" target="_blank">myrecipebook.net</a> והתחבר, ואז חזור לכאן</small>
      </p>
    </div>
  `;
  
  document.getElementById('loginBtn').addEventListener('click', () => {
    chrome.tabs.create({ url: `${CONFIG.API_BASE_URL}/login` });
  });
}

// Handle extract button click
async function handleExtract() {
  // Get URL based on active tab
  let urlToExtract = activeTab === 'current' ? currentPageData.url : pasteUrl.value.trim();
  
  // Validate URL
  if (!urlToExtract || urlToExtract.startsWith('chrome://')) {
    showStatus('error', 'לא ניתן לחלץ מדף זה');
    return;
  }
  
  if (activeTab === 'paste' && !isValidUrl(urlToExtract)) {
    showStatus('error', 'הקישור אינו תקין');
    return;
  }

  // Show extracting state
  mainContent.style.display = 'none';
  tabContentCurrent.style.display = 'none';
  tabContentPaste.style.display = 'none';
  document.querySelector('.tab-switcher').style.display = 'none';
  extractingOverlay.style.display = 'flex';
  
  try {
    // Scrape the URL to get recipe data
    const scrapePayload = {
      url: urlToExtract,
      hints: activeTab === 'current' ? {
        title: currentPageData.title,
        description: currentPageData.description,
        imageUrl: currentPageData.imageUrl,
        source: currentPageData.source
      } : {}
    };
    
    console.log('Scraping URL:', urlToExtract);
    
    const scrapeResponse = await fetch(`${CONFIG.API_BASE_URL}/api/scrape`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      credentials: 'include',
      body: JSON.stringify(scrapePayload)
    });

    console.log('Scrape response status:', scrapeResponse.status);

    if (!scrapeResponse.ok) {
      let errorMessage = 'שגיאה בחילוץ המתכון';
      try {
        const errorData = await scrapeResponse.json();
        errorMessage = errorData.error || errorMessage;
        console.error('API Error:', errorData);
      } catch (e) {
        console.error('Could not parse error response');
      }
      
      if (scrapeResponse.status === 401) {
        showLoginPrompt();
        extractingOverlay.style.display = 'none';
        return;
      }
      
      throw new Error(errorMessage);
    }

    const scrapeData = await scrapeResponse.json();
    
    // Store extracted data
    extractedData = {
      url: urlToExtract,
      title: scrapeData.title || currentPageData.title || 'מתכון חדש',
      description: scrapeData.description || currentPageData.description || '',
      image_path: scrapeData.image_path || '',
      image_url: scrapeData.image_url || currentPageData.imageUrl || '',
      suggested_categories: scrapeData.suggested_categories || [],
      video_embeddable: scrapeData.video_embeddable
    };
    
    // Show preview
    showPreview(extractedData);
    
  } catch (error) {
    console.error('Error extracting recipe:', error);
    
    // Handle different error types
    let errorMessage = 'שגיאה בחילוץ המתכון';
    if (error instanceof TypeError && error.message.includes('Failed to fetch')) {
      errorMessage = 'לא ניתן להתחבר לשרת. בדוק שאתה מחובר לאינטרנט ומחובר לאתר';
      console.error('Network error - check if you are logged in at myrecipebook.net');
    } else if (error.message && error.message !== 'Error') {
      errorMessage = error.message;
    }
    
    showStatus('error', errorMessage);
    
    // Reset to initial state
    extractingOverlay.style.display = 'none';
    document.querySelector('.tab-switcher').style.display = 'flex';
    if (activeTab === 'current') {
      tabContentCurrent.style.display = 'block';
    } else {
      tabContentPaste.style.display = 'block';
    }
    mainContent.style.display = 'block';
  }
}

// Show preview with extracted data
function showPreview(data) {
  // Hide extracting overlay
  extractingOverlay.style.display = 'none';
  
  // Show preview section
  previewSection.style.display = 'block';
  
  // Set image
  const imageUrl = data.image_url || data.image_path;
  if (imageUrl) {
    previewImage.src = imageUrl.startsWith('http') ? imageUrl : `${CONFIG.API_BASE_URL}/static/${imageUrl}`;
    previewImage.style.display = 'block';
    previewImagePlaceholder.style.display = 'none';
    
    // Handle image load error
    previewImage.onerror = () => {
      previewImage.style.display = 'none';
      previewImagePlaceholder.style.display = 'flex';
    };
  } else {
    previewImage.style.display = 'none';
    previewImagePlaceholder.style.display = 'flex';
  }
  
  // Set title
  previewTitle.textContent = data.title;
  
  // Set description
  if (data.description) {
    previewDescription.textContent = data.description.substring(0, 150) + (data.description.length > 150 ? '...' : '');
    previewDescription.style.display = 'block';
  } else {
    previewDescription.style.display = 'none';
  }
  
  // Set source
  const source = detectSource(data.url);
  const sourceLabels = {
    'facebook': '📘 פייסבוק',
    'instagram': '📷 אינסטגרם',
    'tiktok': '🎵 טיקטוק',
    'generic': '🌐 אתר'
  };
  previewSource.textContent = sourceLabels[source] || sourceLabels['generic'];
  
  // Set categories
  if (data.suggested_categories && data.suggested_categories.length > 0) {
    const categoryNames = data.suggested_categories.map(c => c.name).join(', ');
    previewCategories.textContent = `📁 ${categoryNames}`;
    previewCategories.style.display = 'inline';
  } else {
    previewCategories.style.display = 'none';
  }
  
  // Show action buttons
  postExtractActions.style.display = 'block';
}

// Handle save recipe button click
async function handleSaveRecipe() {
  if (!extractedData) {
    showStatus('error', 'אין מידע לשמירה');
    return;
  }

  // Set loading state
  saveBtn.classList.add('loading');
  saveBtn.disabled = true;
  editBtn.disabled = true;
  
  try {
    const categoryIds = extractedData.suggested_categories?.map(c => c.id) || [];
    
    const payload = {
      title: extractedData.title,
      description: extractedData.description,
      url: extractedData.url,
      image_path: extractedData.image_path,
      image_url: extractedData.image_url,
      category_ids: categoryIds,
      source_type: 'link'
    };
    
    console.log('Saving recipe with payload:', payload);
    
    const createResponse = await fetch(`${CONFIG.API_BASE_URL}/api/recipes`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      credentials: 'include',
      body: JSON.stringify(payload)
    });

    console.log('Save response status:', createResponse.status);

    if (!createResponse.ok) {
      let errorMessage = 'שגיאה בשמירת המתכון';
      try {
        const errorData = await createResponse.json();
        errorMessage = errorData.error || errorMessage;
        console.error('Save API Error:', errorData);
      } catch (e) {
        console.error('Could not parse save error response');
      }
      throw new Error(errorMessage);
    }

    const recipeData = await createResponse.json();
    
    // Success!
    showStatus('success', 'המתכון נשמר בהצלחה! 🎉');
    
    // Update buttons
    postExtractActions.innerHTML = `
      <button class="btn btn-success" disabled>
        <span class="btn-icon">✅</span>
        <span class="btn-text">נשמר בהצלחה!</span>
      </button>
      <button class="btn btn-view" id="viewRecipeBtn">
        <span class="btn-icon">👀</span>
        <span class="btn-text">צפה במתכון</span>
      </button>
    `;
    
    document.getElementById('viewRecipeBtn').addEventListener('click', () => {
      chrome.tabs.create({ url: `${CONFIG.API_BASE_URL}/recipe/${recipeData.id}` });
    });

  } catch (error) {
    console.error('Error saving recipe:', error);
    
    // Handle different error types
    let errorMessage = 'שגיאה בשמירת המתכון';
    if (error instanceof TypeError && error.message.includes('Failed to fetch')) {
      errorMessage = 'לא ניתן להתחבר לשרת. בדוק שאתה מחובר לאינטרנט ומחובר לאתר';
    } else if (error.message && error.message !== 'Error') {
      errorMessage = error.message;
    }
    
    showStatus('error', errorMessage);
    saveBtn.classList.remove('loading');
    saveBtn.disabled = false;
    editBtn.disabled = false;
  }
}

// Handle edit recipe button click
function handleEditRecipe() {
  if (!extractedData) {
    showStatus('error', 'אין מידע לעריכה');
    return;
  }
  
  // Build URL with query parameters for the add recipe page
  const params = new URLSearchParams({
    url: extractedData.url,
    title: extractedData.title,
    description: extractedData.description || '',
    image_path: extractedData.image_path || '',
    image_url: extractedData.image_url || '',
    categories: (extractedData.suggested_categories || []).map(c => c.id).join(','),
    from_extension: 'true'
  });
  
  // Open the add recipe page with pre-filled data
  chrome.tabs.create({ url: `${CONFIG.API_BASE_URL}/add?${params.toString()}` });
}

// Show status message
function showStatus(type, message) {
  status.className = `status show ${type}`;
  
  const icon = type === 'success' ? '✅' : '❌';
  status.innerHTML = `<span class="status-icon">${icon}</span>${message}`;
  
  if (type === 'error') {
    setTimeout(() => {
      status.classList.remove('show');
    }, 5000);
  }
}
