#!/usr/bin/env python3
"""
Generate PNG icons from SVG for the Chrome Extension.
Requires: pip install Pillow cairosvg (or just Pillow with manual conversion)
"""

import os
from pathlib import Path

# Icon sizes needed for Chrome Extension
SIZES = [16, 32, 48, 128]

def generate_icons_with_pillow():
    """Generate icons using PIL (simpler, from a base PNG or creating programmatically)"""
    try:
        from PIL import Image, ImageDraw
        
        icons_dir = Path(__file__).parent / "icons"
        icons_dir.mkdir(exist_ok=True)
        
        for size in SIZES:
            # Create a new image with gradient-like background
            img = Image.new('RGBA', (size, size), (0, 0, 0, 0))
            draw = ImageDraw.Draw(img)
            
            # Draw circular gradient background (simplified)
            center = size // 2
            for i in range(center, 0, -1):
                # Gradient from #ff6b6b to #ff8c42
                ratio = i / center
                r = int(255)
                g = int(107 + (140 - 107) * (1 - ratio))
                b = int(107 + (66 - 107) * (1 - ratio))
                draw.ellipse(
                    [center - i, center - i, center + i, center + i],
                    fill=(r, g, b, 255)
                )
            
            # Draw a simple book shape (white)
            margin = size * 0.2
            book_left = margin
            book_right = size - margin
            book_top = margin * 0.8
            book_bottom = size - margin * 0.8
            
            # Book shape
            draw.rounded_rectangle(
                [book_left, book_top, book_right, book_bottom],
                radius=size * 0.05,
                fill=(255, 255, 255, 240)
            )
            
            # Book spine (darker line)
            spine_width = size * 0.06
            draw.rectangle(
                [book_left, book_top, book_left + spine_width, book_bottom],
                fill=(200, 200, 200, 100)
            )
            
            # Draw lines on the book (text representation)
            line_start = book_left + size * 0.15
            line_width = (book_right - book_left) * 0.6
            line_height = max(1, size * 0.03)
            line_color = (255, 107, 107, 180)
            
            for j, y_offset in enumerate([0.25, 0.4, 0.55]):
                y = book_top + (book_bottom - book_top) * y_offset
                width = line_width if j == 0 else line_width * (0.8 if j == 1 else 0.7)
                draw.rounded_rectangle(
                    [line_start, y, line_start + width, y + line_height],
                    radius=line_height / 2,
                    fill=line_color if j == 0 else (220, 220, 220, 255)
                )
            
            # Save the icon
            output_path = icons_dir / f"icon{size}.png"
            img.save(output_path, 'PNG')
            print(f"[OK] Created {output_path}")
        
        print(f"\n[DONE] All icons generated in {icons_dir}")
        return True
        
    except ImportError:
        print("Pillow not installed. Install with: pip install Pillow")
        return False


def generate_icons_with_cairosvg():
    """Generate icons from SVG using cairosvg (higher quality)"""
    try:
        import cairosvg
        
        icons_dir = Path(__file__).parent / "icons"
        svg_path = icons_dir / "icon.svg"
        
        if not svg_path.exists():
            print(f"SVG file not found: {svg_path}")
            return False
        
        for size in SIZES:
            output_path = icons_dir / f"icon{size}.png"
            cairosvg.svg2png(
                url=str(svg_path),
                write_to=str(output_path),
                output_width=size,
                output_height=size
            )
            print(f"[OK] Created {output_path}")
        
        print(f"\n[DONE] All icons generated from SVG in {icons_dir}")
        return True
        
    except (ImportError, OSError) as e:
        print(f"cairosvg not available ({type(e).__name__}). Falling back to Pillow...")
        return False


if __name__ == "__main__":
    print("Generating Chrome Extension icons...\n")
    
    # Try cairosvg first (better quality from SVG), fall back to Pillow
    if not generate_icons_with_cairosvg():
        generate_icons_with_pillow()

