/**
 * Content Script for Recipe Book Web Clipper
 * Runs on Facebook, Instagram, and TikTok to extract content metadata
 */

// Main dish list - used to identify recipe names (same as server)
const MAIN_DISHES = [
  // Combined/compound dishes (highest priority)
  'אורז עם פרגיות', 'אורז עם עוף', 'אורז עם בשר', 'אורז עם ירקות',
  'פסטה עם עוף', 'פסטה עם בשר', 'פסטה ברוטב',
  'פופקורן פירה פטריות', 'פירה פטריות',
  // Meat dishes (high priority)
  'קבב', 'קבבים', 'כיפתה', 'כיפתות', 'קציצות', 'קציצה',
  'שניצל', 'שניצלים', 'פרגיות', 'פרגית', 'חזה עוף', 'שוקי עוף', 'כרעיים',
  'אנטריקוט', 'סטייק', 'צלעות', 'המבורגר', 'המבורגרים',
  'שווארמה', 'כבש', 'טלה', 'כבד עוף', 'כנפיים', 'כנפי עוף',
  'עוף', 'בשר', 'דג',
  // Fish dishes
  'סלמון', 'דניס', 'לברק', 'טונה', 'פילה דג',
  // Asian dishes  
  'סושי', 'נודלס', 'ראמן', 'פאד תאי', 'קארי', 'מוקפץ', 'טריאקי',
  // Standalone dishes
  'שקשוקה', 'פלאפל', 'חריימה', 'קובה', 'לזניה', 'פסטה', 'ספגטי',
  'מרק עוף', 'מרק ירקות', 'מרק', 'נזיד', 'תבשיל',
  'צ\'ולנט', 'חמין', 'מוסקה', 'פריקסה',
  'אורז', 'קוסקוס', 'מגדרה',
  'פיצה', 'בורקס', 'פשטידה', 'קיש', 'חלה', 'לחם',
  // Desserts
  'עוגת גבינה', 'עוגת שוקולד', 'עוגת תפוזים', 'עוגה',
  'עוגיות', 'בראוניז', 'לביבות', 'פנקייק', 'סופגניות',
  // Sides that can be main
  'פירה', 'חומוס', 'טחינה'
];

// Cooking styles that extend dish names
const COOKING_STYLES = [
  'חמוץ מתוק', 'טריאקי', 'בטריאקי', 'ברוטב', 'בשמנת', 'בלימון',
  'צלוי', 'אפוי', 'מטוגן', 'מוקפץ', 'בגריל', 'על האש',
  'בסגנון', 'סיני', 'אסייתי', 'איטלקי', 'מרוקאי', 'הודי',
  'קריספי', 'פריך', 'עסיסי', 'ביתי', 'מושלם'
];

/**
 * Extract dish name from text using the same rules as the server
 */
function extractDishName(text) {
  if (!text || text.length < 3) return null;
  
  // Clean text from emojis, hashtags, RTL markers
  let cleaned = text
    .replace(/#\w+/g, '')
    .replace(/[\u200e\u200f\u202a-\u202e]/g, '')
    .replace(/[\u{1F300}-\u{1F9FF}\u{1FA00}-\u{1FAFF}\u{2700}-\u{27BF}\u{1F600}-\u{1F64F}\u{1F680}-\u{1F6FF}]/gu, '')
    .trim();
  
  // Find main dish in text (prioritize by position - earlier = better)
  function findMainDish(str) {
    const lower = str.toLowerCase();
    let foundDishes = [];
    
    for (const dish of MAIN_DISHES) {
      const pos = lower.indexOf(dish);
      if (pos >= 0) {
        foundDishes.push({ pos, dish, len: dish.length });
      }
    }
    
    if (foundDishes.length === 0) return null;
    
    // Sort by position (first appearance wins) then by length (longer = more specific)
    foundDishes.sort((a, b) => {
      if (a.pos !== b.pos) return a.pos - b.pos;
      return b.len - a.len;  // Prefer longer (more specific) dishes
    });
    
    return foundDishes[0];
  }
  
  // Strategy 0: Check if text STARTS with a dish name (+ optional cooking style)
  const firstPart = cleaned.substring(0, 80);
  const firstDish = findMainDish(firstPart);
  
  if (firstDish && firstDish.pos <= 15) {
    // Dish is near the start - look for cooking style after it
    const afterDish = cleaned.substring(firstDish.pos + firstDish.len, firstDish.pos + firstDish.len + 40);
    
    for (const style of COOKING_STYLES) {
      const stylePos = afterDish.toLowerCase().indexOf(style);
      if (stylePos >= 0 && stylePos <= 15) {
        const fullName = firstDish.dish + ' ' + afterDish.substring(0, stylePos + style.length).trim();
        console.log('Recipe Clipper: Found dish+style:', fullName);
        return fullName.replace(/\s+/g, ' ').trim();
      }
    }
    
    // Check for compound patterns (עם X)
    const compoundMatch = afterDish.match(/^\s*(עם\s+[\u0590-\u05FF]+)/);
    if (compoundMatch) {
      const fullName = firstDish.dish + ' ' + compoundMatch[1];
      console.log('Recipe Clipper: Found compound dish:', fullName);
      return fullName.replace(/\s+/g, ' ').trim();
    }
    
    // Return just the dish name found at start
    console.log('Recipe Clipper: Found dish at start:', firstDish.dish);
    return firstDish.dish;
  }
  
  // Strategy 1: Look for explicit recipe patterns (מתכון ל..., חומרים ל...)
  const recipePatterns = [
    /מתכון\s+(?:ל|של)\s*/i,
    /חומרים\s+ל[-–]?\s*\d*\s*/i,
    /מרכיבים\s+ל[-–]?\s*\d*\s*/i,
    /איך\s+(?:ל)?הכין\s+/i,
    /איך\s+מכינים\s+/i,
    /(?:הכנתי|עשיתי)\s+(?:לכם\s+)?/i
  ];
  
  for (const pattern of recipePatterns) {
    const match = cleaned.match(pattern);
    if (match) {
      const afterPattern = cleaned.substring(match.index + match[0].length, match.index + match[0].length + 60);
      const dishInPattern = findMainDish(afterPattern);
      if (dishInPattern) {
        // Check for compound
        const afterDishInPattern = afterPattern.substring(dishInPattern.pos + dishInPattern.len, dishInPattern.pos + dishInPattern.len + 30);
        const compound = afterDishInPattern.match(/^\s*(עם\s+[\u0590-\u05FF]+)/);
        if (compound) {
          const result = dishInPattern.dish + ' ' + compound[1];
          console.log('Recipe Clipper: Found via pattern:', result);
          return result.replace(/\s+/g, ' ').trim();
        }
        console.log('Recipe Clipper: Found via pattern:', dishInPattern.dish);
        return dishInPattern.dish;
      }
    }
  }
  
  // Strategy 2: Find any main dish in the full text
  const anyDish = findMainDish(cleaned);
  if (anyDish) {
    // Look for compound
    const afterAnyDish = cleaned.substring(anyDish.pos + anyDish.len, anyDish.pos + anyDish.len + 30);
    const compoundAny = afterAnyDish.match(/^\s*(עם\s+[\u0590-\u05FF]+)/);
    if (compoundAny) {
      const result = anyDish.dish + ' ' + compoundAny[1];
      console.log('Recipe Clipper: Found compound in text:', result);
      return result.replace(/\s+/g, ' ').trim();
    }
    console.log('Recipe Clipper: Found dish in text:', anyDish.dish);
    return anyDish.dish;
  }
  
  return null;
}

/**
 * Clean and format title - remove platform names, extract dish name
 */
function cleanTitle(rawTitle, description = '') {
  if (!rawTitle) return null;
  
  // Remove platform suffixes
  let title = rawTitle
    .replace(/\s*[|\-–]\s*(Facebook|Instagram|TikTok|פייסבוק|אינסטגרם|טיקטוק).*$/i, '')
    .replace(/^(Facebook|Instagram|TikTok|Watch|Reels?)[\s:|]*/i, '')
    .trim();
  
  // If title is still generic, try to extract from description
  if (!title || title.length < 5 || /^(Facebook|Watch|Video|Reel)$/i.test(title)) {
    if (description) {
      // Try first line of description
      const firstLine = description.split(/[\n\r]+/)[0]?.trim();
      if (firstLine && firstLine.length > 5 && firstLine.length < 100) {
        title = firstLine;
      }
    }
  }
  
  // Try to extract a specific dish name from title or description
  const dishFromTitle = extractDishName(title);
  const dishFromDesc = description ? extractDishName(description) : null;
  
  // Prefer dish from description if title dish is generic
  if (dishFromDesc && (!dishFromTitle || dishFromTitle.length < dishFromDesc.length)) {
    return dishFromDesc;
  }
  
  if (dishFromTitle) {
    return dishFromTitle;
  }
  
  // If no dish found, return cleaned first line (max 80 chars)
  if (title.length > 80) {
    title = title.substring(0, 80) + '...';
  }
  
  return title || null;
}

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getPageContent') {
    const content = extractPageContent();
    sendResponse(content);
  }
  return true; // Keep the message channel open for async response
});

/**
 * Main function to extract content based on the current site
 */
function extractPageContent() {
  const url = window.location.href;
  const hostname = window.location.hostname;
  
  let result = {
    url: url,
    title: document.title,
    description: '',
    imageUrl: '',
    source: 'generic',
    confidence: 'low'
  };
  
  try {
    if (hostname.includes('facebook.com')) {
      result = extractFacebookContent(result);
    } else if (hostname.includes('instagram.com')) {
      result = extractInstagramContent(result);
    } else if (hostname.includes('tiktok.com')) {
      result = extractTikTokContent(result);
    } else {
      result = extractGenericContent(result);
    }
  } catch (error) {
    console.error('Recipe Clipper: Error extracting content', error);
  }
  
  return result;
}

/**
 * Extract content from Facebook
 */
function extractFacebookContent(result) {
  result.source = 'facebook';
  
  // Check if this is a reel/video page
  const isReelPage = window.location.href.includes('/reel/');
  const isVideoPage = window.location.href.includes('/videos/') || window.location.href.includes('/watch');
  
  // Method 1: Get video/reel URL if watching specific content
  const urlMatch = window.location.href.match(/\/(?:watch\/?\?v=|videos\/|reel\/)(\d+)/);
  if (urlMatch) {
    result.url = window.location.href;
    result.confidence = 'high';
  }
  
  // Method 2: Extract post/reel description - THIS IS WHERE THE RECIPE NAME USUALLY IS
  let postText = '';
  
  // Facebook Reel specific selectors (2024 structure)
  const reelDescriptionSelectors = [
    // Reel caption/description
    'div[data-pagelet="ReelsSurfacePlayer"] span[dir="auto"]',
    'div[data-pagelet*="Reel"] span[dir="auto"]',
    // Video description area
    '[aria-label*="תיאור"], [aria-label*="description"]',
    // Common text containers in reels
    'div.x1lliihq.x6ikm8r.x10wlt62.x1n2onr6 span[dir="auto"]',
    // Post text in feed
    'div[data-ad-preview="message"]',
    'div[data-ad-comet-preview="message"]',
    // Generic text containers
    'div.xdj266r span[dir="auto"]',
    'div.x11i5rnm span[dir="auto"]',
    // Expanded text
    'div[dir="auto"][style*="-webkit-line-clamp"]',
    'div[dir="auto"].x1iorvi4',
    // Any visible text block
    '[role="article"] div[dir="auto"]',
    'span.x193iq5w[dir="auto"]'
  ];
  
  // Try each selector and get the best text
  for (const selector of reelDescriptionSelectors) {
    try {
      const elements = document.querySelectorAll(selector);
      for (const el of elements) {
        const text = el.textContent.trim();
        // Skip very short texts, buttons, or navigation elements
        if (text.length > 10 && text.length > postText.length && !isNavigationText(text)) {
          // Check if element is visible
          const rect = el.getBoundingClientRect();
          if (rect.width > 0 && rect.height > 0) {
            postText = text;
          }
        }
      }
    } catch (e) {
      // Skip invalid selectors
    }
  }
  
  // Method 3: Try to get text from the page's main content area
  if (!postText || postText.length < 20) {
    // Get all spans with Hebrew text (likely recipe descriptions)
    const allSpans = document.querySelectorAll('span[dir="auto"]');
    for (const span of allSpans) {
      const text = span.textContent.trim();
      // Look for text that looks like a recipe (has Hebrew, reasonable length)
      if (text.length > 20 && text.length < 2000 && /[\u0590-\u05FF]/.test(text)) {
        if (!isNavigationText(text) && text.length > postText.length) {
          postText = text;
        }
      }
    }
  }
  
  // Method 4: Extract from og:description (sometimes has the caption)
  const ogDesc = document.querySelector('meta[property="og:description"]');
  if (ogDesc && ogDesc.content && ogDesc.content.length > postText.length) {
    const desc = ogDesc.content;
    if (!desc.includes('Facebook') && desc.length > 20) {
      postText = desc;
    }
  }
  
  // Store description
  if (postText) {
    result.description = postText.substring(0, 500);
  }
  
  // Method 5: Extract title using the dish extraction rules
  // Try to find a specific dish name from the post text
  const extractedDish = cleanTitle(null, postText);
  
  if (extractedDish && extractedDish.length >= 3) {
    result.title = extractedDish;
    console.log('Recipe Clipper: Extracted dish name:', extractedDish);
  } else if (postText) {
    // Fallback: first line of post text
    const lines = postText.split(/[\n\r]+/).filter(line => line.trim().length > 3);
    if (lines.length > 0) {
      let titleLine = lines[0].trim()
        .replace(/^[\s🍳🥘🍲🍝🍕🍔🥗🍰🎂\u{1F300}-\u{1F9FF}]+/u, '')
        .trim();
      
      if (titleLine.length > 3 && titleLine.length < 100) {
        result.title = titleLine;
      } else {
        result.title = postText.substring(0, 80).replace(/\n/g, ' ').trim();
        if (postText.length > 80) result.title += '...';
      }
    }
  }
  
  // Method 6: Fallback to og:title if no good title found
  if (!result.title || result.title === 'Facebook' || result.title.includes('| Facebook')) {
    const ogTitle = document.querySelector('meta[property="og:title"]');
    if (ogTitle && ogTitle.content && !ogTitle.content.includes('Facebook')) {
      // Try to extract dish from og:title too
      const dishFromOg = cleanTitle(ogTitle.content, result.description);
      result.title = dishFromOg || ogTitle.content;
    }
  }
  
  // Method 7: Try to get author name
  let authorName = '';
  const authorSelectors = [
    'a[role="link"] strong',
    'h2 a span strong',
    'span.x193iq5w strong',
    'a[aria-label] strong'
  ];
  
  for (const selector of authorSelectors) {
    const authorEl = document.querySelector(selector);
    if (authorEl && authorEl.textContent.trim()) {
      const name = authorEl.textContent.trim();
      if (name.length > 2 && name.length < 50 && !name.includes('Facebook')) {
        authorName = name;
        result.author = authorName;
        break;
      }
    }
  }
  
  // If we have author but generic/no title, create a combined title
  if (authorName && (!result.title || result.title.length < 5)) {
    if (result.description) {
      const preview = result.description.substring(0, 60).replace(/\n/g, ' ');
      result.title = `${authorName}: ${preview}`;
    } else {
      result.title = `מתכון מאת ${authorName}`;
    }
  }
  
  // Get image
  const ogImage = document.querySelector('meta[property="og:image"]');
  if (ogImage && ogImage.content) {
    result.imageUrl = ogImage.content;
  }
  
  // Try to find video thumbnail
  const videoThumb = document.querySelector('video[poster]');
  if (videoThumb && videoThumb.poster) {
    result.imageUrl = videoThumb.poster;
  }
  
  console.log('Facebook extraction result:', { 
    title: result.title, 
    extractedDish: extractedDish,
    desc: result.description?.substring(0, 100) 
  });
  
  return result;
}

/**
 * Check if text is navigation/UI text that should be skipped
 */
function isNavigationText(text) {
  const navPatterns = [
    /^(like|comment|share|שיתוף|תגובה|אהבתי|עוד|see more|ראה עוד|הצג עוד)$/i,
    /^(follow|עקוב|הירשם|subscribe)$/i,
    /^\d+[KMk]?\s*(likes?|comments?|shares?|צפיות|תגובות|שיתופים)/i,
    /^(reels?|stories?|watch|video)/i,
    /^(home|בית|explore|חקור|notifications|התראות)$/i
  ];
  
  for (const pattern of navPatterns) {
    if (pattern.test(text.trim())) {
      return true;
    }
  }
  
  return false;
}

/**
 * Extract content from Instagram
 */
function extractInstagramContent(result) {
  result.source = 'instagram';
  
  // Check if on a specific post/reel page
  const isPostPage = /\/(p|reel)\/[\w-]+/.test(window.location.pathname);
  
  if (isPostPage) {
    result.confidence = 'high';
    result.url = window.location.href;
  }
  
  // Get description from og:description or post caption
  const ogDesc = document.querySelector('meta[property="og:description"]');
  if (ogDesc && ogDesc.content) {
    result.description = ogDesc.content;
  }
  
  // Try to find caption in the page
  const captionSelectors = [
    'h1._ap3a._aaco._aacu._aacx._aad7._aade',
    'div._a9zs span',
    'article span[dir="auto"]'
  ];
  
  for (const selector of captionSelectors) {
    const caption = document.querySelector(selector);
    if (caption && caption.textContent.trim()) {
      const text = caption.textContent.trim();
      if (text.length > 10) {
        result.description = text.substring(0, 500);
        break;
      }
    }
  }
  
  // Extract title using dish rules
  const ogTitle = document.querySelector('meta[property="og:title"]');
  const rawTitle = ogTitle?.content || '';
  
  // Try to extract a specific dish name
  const extractedDish = cleanTitle(rawTitle, result.description);
  
  if (extractedDish && extractedDish.length >= 3) {
    result.title = extractedDish;
  } else if (rawTitle && !rawTitle.includes('Instagram')) {
    result.title = rawTitle;
  } else if (result.description) {
    result.title = result.description.substring(0, 100) + (result.description.length > 100 ? '...' : '');
  }
  
  // Get image
  const ogImage = document.querySelector('meta[property="og:image"]');
  if (ogImage && ogImage.content) {
    result.imageUrl = ogImage.content;
  }
  
  console.log('Instagram extraction result:', { title: result.title, extractedDish });
  
  return result;
}

/**
 * Extract content from TikTok
 */
function extractTikTokContent(result) {
  result.source = 'tiktok';
  
  // Check if on a specific video page
  const isVideoPage = /\/@[\w.]+\/video\/\d+/.test(window.location.pathname) || 
                      window.location.pathname.includes('/video/');
  
  if (isVideoPage) {
    result.confidence = 'high';
    result.url = window.location.href;
  }
  
  // Get description from og:description
  const ogDesc = document.querySelector('meta[property="og:description"]');
  if (ogDesc && ogDesc.content) {
    result.description = ogDesc.content;
  }
  
  // Try to find video description in the page
  const descSelectors = [
    '[data-e2e="browse-video-desc"]',
    '[data-e2e="video-desc"]',
    '.tiktok-j2a19r-SpanText',
    'h1[data-e2e="browse-video-desc"]'
  ];
  
  for (const selector of descSelectors) {
    const descEl = document.querySelector(selector);
    if (descEl && descEl.textContent.trim()) {
      const text = descEl.textContent.trim();
      result.description = text.substring(0, 500);
      break;
    }
  }
  
  // Extract title using dish rules
  const ogTitle = document.querySelector('meta[property="og:title"]');
  const rawTitle = ogTitle?.content || '';
  
  // Try to extract a specific dish name from description (usually has the recipe)
  const extractedDish = cleanTitle(rawTitle, result.description);
  
  if (extractedDish && extractedDish.length >= 3) {
    result.title = extractedDish;
  } else if (rawTitle && rawTitle !== 'TikTok') {
    result.title = rawTitle.substring(0, 100);
  } else if (result.description) {
    result.title = result.description.substring(0, 100) + (result.description.length > 100 ? '...' : '');
  }
  
  // Get author name
  let authorName = '';
  const authorSelectors = [
    '[data-e2e="browse-username"]',
    '[data-e2e="video-author-uniqueid"]',
    'h3[data-e2e="video-author-uniqueid"]'
  ];
  
  for (const selector of authorSelectors) {
    const authorEl = document.querySelector(selector);
    if (authorEl && authorEl.textContent.trim()) {
      authorName = authorEl.textContent.trim();
      result.author = authorName;
      break;
    }
  }
  
  // Get image/thumbnail
  const ogImage = document.querySelector('meta[property="og:image"]');
  if (ogImage && ogImage.content) {
    result.imageUrl = ogImage.content;
  }
  
  console.log('TikTok extraction result:', { title: result.title, extractedDish, author: authorName });
  
  return result;
}

/**
 * Extract content from generic pages
 */
function extractGenericContent(result) {
  result.source = 'generic';
  
  // Get og:description
  const ogDesc = document.querySelector('meta[property="og:description"]') || 
                 document.querySelector('meta[name="description"]');
  if (ogDesc && ogDesc.content) {
    result.description = ogDesc.content;
  }
  
  // Get og:image
  const ogImage = document.querySelector('meta[property="og:image"]');
  if (ogImage && ogImage.content) {
    result.imageUrl = ogImage.content;
  }
  
  // Try to find recipe schema first (most reliable)
  const schemaScripts = document.querySelectorAll('script[type="application/ld+json"]');
  for (const script of schemaScripts) {
    try {
      const data = JSON.parse(script.textContent);
      const recipe = findRecipeInSchema(data);
      if (recipe) {
        result.title = recipe.name || result.title;
        result.description = recipe.description || result.description;
        if (recipe.image) {
          result.imageUrl = Array.isArray(recipe.image) ? recipe.image[0] : 
                           (typeof recipe.image === 'object' ? recipe.image.url : recipe.image);
        }
        result.confidence = 'high';
        console.log('Generic extraction: Found recipe schema:', result.title);
        return result;
      }
    } catch (e) {
      // Invalid JSON, skip
    }
  }
  
  // No schema found - extract title using dish rules
  const ogTitle = document.querySelector('meta[property="og:title"]');
  const rawTitle = ogTitle?.content || document.title || '';
  
  // Try to extract a specific dish name
  const extractedDish = cleanTitle(rawTitle, result.description);
  
  if (extractedDish && extractedDish.length >= 3) {
    result.title = extractedDish;
  } else {
    result.title = rawTitle;
  }
  
  console.log('Generic extraction result:', { title: result.title, extractedDish });
  
  return result;
}

/**
 * Find Recipe schema in JSON-LD data
 */
function findRecipeInSchema(data) {
  if (!data) return null;
  
  if (Array.isArray(data)) {
    for (const item of data) {
      const result = findRecipeInSchema(item);
      if (result) return result;
    }
    return null;
  }
  
  if (data['@type'] === 'Recipe') {
    return data;
  }
  
  if (data['@graph']) {
    return findRecipeInSchema(data['@graph']);
  }
  
  return null;
}

/**
 * Check if element is in viewport
 */
function isElementInViewport(el) {
  if (!el) return false;
  
  const rect = el.getBoundingClientRect();
  const windowHeight = window.innerHeight || document.documentElement.clientHeight;
  const windowWidth = window.innerWidth || document.documentElement.clientWidth;
  
  // Check if element is at least partially visible
  const vertInView = (rect.top <= windowHeight) && ((rect.top + rect.height) >= 0);
  const horInView = (rect.left <= windowWidth) && ((rect.left + rect.width) >= 0);
  
  // Also check if element has reasonable size (not hidden)
  const hasSize = rect.width > 10 && rect.height > 10;
  
  return vertInView && horInView && hasSize;
}

// Log that content script is loaded
console.log('Recipe Clipper: Content script loaded on', window.location.hostname);



